//------------------------------------------------------------------------------
// File: client_cmddisable.cs
// This file contains script functions for disabling and enabling command 
// buttons
// Author: Matthew Rudge
//------------------------------------------------------------------------------

//-Data-------------------------------------------------------------------------
$CommandsDisabledCount = 0;
//------------------------------------------------------------------------------

//-Functions--------------------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
//! Adds a command to the list of commands that are disabled
//! \param %cmd Command string to disable
////////////////////////////////////////////////////////////////////////////////
function csDisableCommand(%cmd)
{
   // Already disabled
   if(csIsCommandDisabled(%cmd)) {
      return;
   }
   
   // Disable command
   $CommandsDisabled[$CommandsDisabledCount] = %cmd;
   $CommandsDisabledCount++;
}

////////////////////////////////////////////////////////////////////////////////
//! Removes a command from the list of commands that are disabled
//! \param %cmd Command string to enable
////////////////////////////////////////////////////////////////////////////////
function csEnableCommand(%cmd)
{
   // Find command
   %found = false;
   for(%i = 0; %i < $CommandsDisabledCount; %i++) {
      if($CommandsDisabled[%i] $= %cmd) {
         %found = true;
         break;
      }
   }
   
   // if the command could not be found, it's already enabled
   if (%found == false)
   {
      return;
   }
   
   // Remove command
   for(; %i < $CommandsDisabledCount - 1; %i++) {
      $CommandsDisabled[%i] = $CommandsDisabled[%i + 1];
   }
   
   // Decrement count
   $CommandsDisabledCount--;
}

////////////////////////////////////////////////////////////////////////////////
//! Checks to see if a command has currently been disabled
//! \pararm %cmd Command string to check
////////////////////////////////////////////////////////////////////////////////
function csIsCommandDisabled(%cmd)
{
   // Find command
   for(%i = 0; %i < $CommandsDisabledCount; %i++) {
      if($CommandsDisabled[%i] $= %cmd) {
         return true;
      }
   }
   return false;
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the current number of disabled commands
//! \retval int Number of disabled commands
////////////////////////////////////////////////////////////////////////////////
function csGetDisabledCommandCount()
{
   return $CommandsDisabledCount;
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the disabled command located at the n-th index
//! \pararm %index Index of disabled command
//! \retval string Disabled command string
////////////////////////////////////////////////////////////////////////////////
function csGetDisabledCommand(%index)
{
   return $CommandsDisabled[%index];
}

////////////////////////////////////////////////////////////////////////////////
//! Clears all disabled commands
////////////////////////////////////////////////////////////////////////////////
function csClearDisabledCommands()
{
   $CommandsDisabledCount = 0;
}
//------------------------------------------------------------------------------

// saves the command enabled and disabled states
function csSaveToFile()
{
   // save the number of disabled commands
   %count = csGetDisabledCommandCount();
   slgSaveInt(%count);
   
   // save out the individual disabled commands
   for (%index = 0; %index < %count; %index++)
   {
      %command = csGetDisabledCommand(%index);
      slgSaveString(%command);
   }
   
   // save out command upgrades
   for (%index = 0; %index < $CSUP_COUNT; %index++)
   {
      %upgrade = $CSUP_VALUES[%index];
      slgSaveBool(%upgrade);
      if (%upgrade == true)
      {
         %value = $CSUP_VALUES_DATA[%index];
         slgSaveString(%value);
      }
   }
}

// loads the command enabled and disabled states
function csLoadFromFile()
{
   // clear the disabled command state
   csClearDisabledCommands();
   
   // load the individual disabled commands
   %count = slgLoadInt();   
   for (%index = 0; %index < %count; %index++)
   {
      %command = slgLoadString();
      csDisableCommand(%command);
   }
   
   // clear the command upgrades
   %client = ClientGroup.getObject(0);
   csResetCmdUpgrades(%client);
   
   // load the command upgrades
   for (%index = 0; %index < $CSUP_COUNT; %index++)
   {
      %upgrade = slgLoadBool();
      if (%upgrade == true)
      {
         %value = slgLoadString();
         csSetCmdUpgrade(%client, %index, %value);
      }
   }
   
   return true;
}


// End client_cmddisable.cs